/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/********************************************************************/
/* File seq_pars.c                                                  */
/* provides a function to parse a transient description             */
/********************************************************************/

#include <string.h>
#include "xseqbool.h"
#include "xseqpars.h"
#include <xtypedef.h>
#include <xutil.h> 

/* need to have a global string buffer and a global pointer to the  */
/* current sequencer descriptor since yyparse() must not have       */
/* parameters                                                       */

char yy_sequence_buffer[YYSEQBUFSIZE];
int  yybufptr;
char yy01context;
int yy_seq__syntax_error;
char yyerrflag;
int seq_tran_count;
bx_handletype BX_GLOBAL_handle;


bx_errtype yaccerr;

sequencer_descriptor * yy_seq__seq_descr;

sequencer_descriptor sequencer_list; /* [NO_OF_SEQUENCERS]; */

/* some extern definitions .. */
extern long yy_seq_maxdepth;

extern int yy_seq_parse();


/********************************************************************/
/* Function error_msg()                                             */
/* This function is called within all other modules to report       */
/* errors. Adapt the body to your system specs!                     */
/********************************************************************/

/*void error_msg(err_string)
       char * err_string;
{ */
/*   sprintf(yyerrbuf,err_string); */
   /*printf("ERROR: %s\n", err_string);*/
/*} */        


/********************************************************************/
/* Function parse_sequence_description()                            */
/* parses the given description string and writes the parse result  */
/* to the sequencer ram map which is linked to the sequencer        */
/* descriptor named 'seq_name'                                      */
/********************************************************************/

bx_errtype ParseSequenceDescription(
  bx_handletype handle
)
{
  bx_errtype status = BX_EFW_OK;

  yy_seq__seq_descr = &sequencer_list; 
  
  yy01context = 0;     /* '0' and '1' are interpreted as decimals */
  yybufptr = 0;
  yy_seq__syntax_error = 0;
  yy_seq_maxdepth = YYMAXDEPTH;
  seq_tran_count = 0;
  
  /* Problem: 
     Code for yy_seq_parse() is generated by yacc, so
     we cannot pass a handle to it. So we use a global handle
     variable here.
     Caution: 
     Replace this variable by a real OS semaphor,
     if you want the CAPI to become reentrant (parser is not reentrant) !
  */
  
  BX_GLOBAL_handle=handle;
  yy_seq_parse();
  BX_GLOBAL_handle=0xffffffff; /* invalid */
  
  if(yy_seq__syntax_error && (yaccerr == BX_EFW_OK))
    status = BX_EFW_FUNC;
  else
    status = yaccerr;

  return status;
}

/********************************************************************/
/* Function init_sequencer_descriptor()                             */
/* This function initialises the sequencer descriptor               */
/* (allocates the ram arrays where the parser writes its            */
/* result to etc.) This function *must* be called before any        */
/* transients are parsed!                                           */
/********************************************************************/

bx_errtype InitSequencerDescriptor(
  bx_charptrtype sequencer_name,
  sequencer_descriptor * seq_descr)
{
  int ram_width;
  int ram_depth;
  int i, states, s, ld_s;

  yaccerr = BX_EFW_OK; /* clear the errors */

  /* copy sequencer name, states, input and output params  */

  sequencer_list.name    = sequencer_name; /* seq_descr -> name; */
  sequencer_list.states  = seq_descr -> states;
  sequencer_list.inputs  = seq_descr -> inputs;
  sequencer_list.outputs = seq_descr -> outputs;
  sequencer_list.input_tokens  = seq_descr -> input_tokens;
  sequencer_list.output_tokens = seq_descr -> output_tokens;
  sequencer_list.input_tokenlist = seq_descr -> input_tokenlist;
  sequencer_list.output_tokenlist = seq_descr -> output_tokenlist;
  sequencer_list.state_tokenlist = seq_descr -> state_tokenlist;

  sequencer_list.default_xcond=
    BestXMemMalloc((seq_descr->states) * sizeof(boolmap *));
  if (sequencer_list.default_xcond==NULL)
  {
    return (BX_E_HOST_MEM_FULL);
  }
  
  sequencer_list.accumul_xcond=
    BestXMemMalloc((seq_descr->states) * sizeof(boolmap *));
  if (sequencer_list.accumul_xcond==NULL)
  {
    return (BX_E_HOST_MEM_FULL);
  }

  for(i=0; i < seq_descr -> states; i++)
  {
    sequencer_list.default_xcond[i] = 
      (boolmap *)alloc_boolmap(seq_descr -> inputs, 254);
    sequencer_list.accumul_xcond[i] = 
      (boolmap *)alloc_boolmap(seq_descr -> inputs, 254);
  }
   

  /* allocate ram array */
  /* for a sequencer with s states, i inputs and o outputs we need */
  /* a ram array of depth s*2^i and width ld(s) + o                */

  states = seq_descr -> states;
  s      = seq_descr -> states;
  ld_s   = 0;

  while((s & 0x01) == 0)
  {
    s = s >> 1;
    ld_s++;
  }
   
  /* the depth of the sequencer ram lin lines */

  ram_depth = (1 << (seq_descr -> inputs)) * states;

  /* the width of the sequencer ram in bits */

  ram_width = ld_s + seq_descr -> outputs;
  
  /* transform ram width into number of chars per ram line */

  if(ram_width % 8)
  {
    ram_width = ram_width / 8 + 1;
  }
  else
  {
    ram_width = ram_width / 8;
  }
  
  sequencer_list.ram_array_ptr=BestXMemMalloc(ram_depth *ram_width);
  if (sequencer_list.ram_array_ptr==NULL)
  {
    return (BX_E_HOST_MEM_FULL);
  }
  
  /* clear ram array (this is essential !!!!) */
  for(i=0; i < ram_depth * ram_width; i++)
    sequencer_list.ram_array_ptr[i] = (char)0x00;

  return BX_E_OK;
}
 
/********************************************************************/
/* Function free_sequencer_descriptor()                             */
/* this function deallocates all malloc'd memory which is linked    */
/* to the sequencer descriptor                                      */
/********************************************************************/

void FreeSequencerDescriptor()
{
  int i;                                                

  for(i=0; i < sequencer_list.states; i++)
  {
    dealloc_boolmap(sequencer_list.default_xcond[i]);
    dealloc_boolmap(sequencer_list.accumul_xcond[i]);
  }

  BestXMemFree((void**)&(sequencer_list.default_xcond));
  BestXMemFree((void**)&(sequencer_list.accumul_xcond));
  BestXMemFree((void**)&(sequencer_list.ram_array_ptr));

  sequencer_list.name = NULL; /* indicates that this entry is empty */
}


/********************************************************************/
/* Function get_sequencer_param()                                   */
/* This function returns parameter values from the current sequencer*/
/*                                                                  */
/********************************************************************/

bx_errtype GetSequencerParam(bx_int32 parameter, bx_int32ptr value)
{
  int states, s, ld_s, ram_depth, ram_width;

  states = sequencer_list.states;
  s      = sequencer_list.states;
  ld_s   = 0;

  while((s & 0x01) == 0)
  {
    s = s >> 1;
    ld_s++;
  }
   
  /* the depth of the sequencer ram in lines */

  ram_depth = (1 << (sequencer_list.inputs)) * states;

  /* the width of the sequencer ram in bits */

  ram_width = ld_s + sequencer_list.outputs;
  
  /* transform ram width into number of chars per ram line */

  if(ram_width % 8)
    ram_width = ram_width / 8 + 1;
  else
    ram_width = ram_width / 8;

  switch(parameter)
  {
   case NO_OF_STATES:   
    *value = (sequencer_list.states);
    break;
   case NO_OF_INPUTS:   
    *value = (sequencer_list.inputs);
    break;
   case NO_OF_OUTPUTS:  
    *value = (sequencer_list.outputs);
    break;
   case SEQ_RAM_WIDTH: 
    *value = (ram_width);
    break;
   case SEQ_RAM_DEPTH:  
    *value = (ram_depth);
    break;
   default: *value = 0;
    break;
  }   

  return BX_EFW_OK;
}


/********************************************************************/
/* Function read_sequencer_ramline()                                */
/* This function reads a line from the ram of the current sequencer */
/*                                                                  */
/********************************************************************/

bx_errtype ReadSequencerRamline(
  bx_int32 line, 
  bx_int32ptr mem_lo,
  bx_int32ptr mem_hi)
{
   
  int j,  ram_width;
  int states, s, ld_s;

  bx_int32 pword_lo,pword_hi,ram_depth;



  states = sequencer_list.states;
  s      = sequencer_list.states;
  ld_s   = 0;

  while((s & 0x01) == 0)
  {
    s = s >> 1;
    ld_s++;
  }

  /* the depth of the sequencer ram in lines */

  ram_depth = (1 << (sequencer_list.inputs)) * states;

  if((line >= ram_depth) || (line < 0))
  {
    /*error_msg("read_sequencer_ramline: index out of bounds");*/
    return BX_EFW_FUNC;
  }

  /* the width of the sequencer ram in bits */

  ram_width = ld_s + sequencer_list.outputs;
  
  /* transform ram width into number of chars per ram line */

  if(ram_width % 8)
    ram_width = ram_width / 8 + 1;
  else
    ram_width = ram_width / 8;

     
  pword_lo = 0x00000000;
  pword_hi = 0x00000000;

  for(j=0; j < ram_width; j++)
    if(j < 4)
      pword_lo |= (((char)(sequencer_list.ram_array_ptr[line * ram_width + j])&0xFF) << (8 * j));
    else
      pword_hi |= (((char)(sequencer_list.ram_array_ptr[line * ram_width + j])&0xFF) << (8 * (j-4)));

  *mem_hi = pword_hi;
  *mem_lo = pword_lo;

  return BX_EFW_OK;

}
